# dialog.py
# Definisce la finestra di dialogo per la scelta dello stile e del percorso di output.

from qgis.PyQt.QtWidgets import (
    QDialog, QVBoxLayout, QComboBox, QLabel, QDialogButtonBox,
    QLineEdit, QPushButton, QHBoxLayout, QFileDialog
)

class StyleDialog(QDialog):
    """
    Una finestra di dialogo che permette all'utente di scegliere le opzioni
    di stile e il percorso di salvataggio per il nuovo layer.
    """
    def __init__(self, fields, parent=None):
        """
        Costruttore.
        :param fields: Una lista di nomi di campi da mostrare nei menu a tendina.
        """
        super(StyleDialog, self).__init__(parent)
        self.setWindowTitle("Impostazioni di Importazione e Stile")
        self.setMinimumWidth(400)

        # Layout principale
        layout = QVBoxLayout(self)

        # --- Sezione Percorso di Salvataggio ---
        self.output_label = QLabel("Salva il layer GeoPackage in:", self)
        
        # Layout orizzontale per casella di testo e pulsante
        output_layout = QHBoxLayout()
        self.output_path_edit = QLineEdit(self)
        self.output_path_edit.setPlaceholderText("Seleziona un percorso per il file .gpkg...")
        self.browse_button = QPushButton("Sfoglia...", self)
        self.browse_button.clicked.connect(self.select_output_path)
        
        output_layout.addWidget(self.output_path_edit)
        output_layout.addWidget(self.browse_button)

        layout.addWidget(self.output_label)
        layout.addLayout(output_layout)
        layout.addSpacing(20)

        # --- Sezione Stile ---
        self.label_label = QLabel("Mostra etichette usando il campo:", self)
        self.label_combo = QComboBox(self)
        
        self.category_label = QLabel("Categorizza simbologia usando il campo:", self)
        self.category_combo = QComboBox(self)
        
        none_option = "Nessuna"
        self.label_combo.addItem(none_option)
        self.category_combo.addItem(none_option)
        
        self.label_combo.addItems(fields)
        self.category_combo.addItems(fields)

        layout.addWidget(self.label_label)
        layout.addWidget(self.label_combo)
        layout.addSpacing(15)
        layout.addWidget(self.category_label)
        layout.addWidget(self.category_combo)

        # --- Pulsanti Finali ---
        self.button_box = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel, self)
        self.button_box.accepted.connect(self.accept)
        self.button_box.rejected.connect(self.reject)
        
        layout.addWidget(self.button_box)
        self.setLayout(layout)

    def select_output_path(self):
        """Apre una finestra di dialogo per salvare il file GeoPackage."""
        path, _ = QFileDialog.getSaveFileName(self, "Salva Layer GeoPackage", "", "GeoPackage (*.gpkg)")
        if path:
            self.output_path_edit.setText(path)

    def get_output_path(self):
        """Restituisce il percorso di output scelto."""
        return self.output_path_edit.text()

    def get_label_field(self):
        """Restituisce il campo scelto per le etichette."""
        return self.label_combo.currentText()

    def get_category_field(self):
        """Restituisce il campo scelto per la categorizzazione."""
        return self.category_combo.currentText()
    
    def accept(self):
        """Controlla che sia stato scelto un percorso prima di chiudere."""
        if not self.get_output_path():
            self.output_path_edit.setFocus()
            self.output_path_edit.setStyleSheet("border: 1px solid red;")
            return # Impedisce la chiusura della finestra
        super().accept()
