# collector_importer_plugin.py
# Questo è il cuore del plugin.

import os
import zipfile
import tempfile
import json
import csv
import random
import shutil

from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, QVariant
from qgis.PyQt.QtWidgets import QAction, QFileDialog
from qgis.PyQt.QtGui import QColor, QIcon
from qgis.core import (
    QgsProject,
    QgsPointXY,
    QgsGeometry,
    QgsFeature,
    QgsVectorLayer,
    QgsField,
    QgsWkbTypes,
    QgsCoordinateReferenceSystem,
    QgsFields,
    Qgis,
    QgsCategorizedSymbolRenderer,
    QgsSymbol,
    QgsRendererCategory,
    QgsVectorLayerSimpleLabeling,
    QgsPalLayerSettings,
    QgsAction,
    QgsVectorFileWriter
)
from qgis.utils import iface

# Importa le risorse (per l'icona)
from .resources import *
# Importa la nuova finestra di dialogo
from .dialog import StyleDialog


class CollectorImporterPlugin:
    """Implementazione del plugin QGIS."""

    def __init__(self, iface):
        """Costruttore."""
        self.iface = iface
        self.plugin_dir = os.path.dirname(__file__)
        self.actions = []
        self.menu = "&Collector Importer"
        self.toolbar = self.iface.addToolBar("CollectorImporterPlugin")
        self.toolbar.setObjectName("CollectorImporterPlugin")

    def tr(self, message):
        """Traduce una stringa."""
        return QCoreApplication.translate('CollectorImporterPlugin', message)

    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Aggiunge un'azione al menu e alla barra degli strumenti."""
        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)
        if status_tip is not None:
            action.setStatusTip(status_tip)
        if whats_this is not None:
            action.setWhatsThis(whats_this)
        if add_to_toolbar:
            self.toolbar.addAction(action)
        if add_to_menu:
            self.iface.addPluginToMenu(self.menu, action)
        self.actions.append(action)
        return action

    def initGui(self):
        """Crea l'interfaccia grafica del plugin."""
        icon_path = ':/plugins/collector_importer/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'Importa Rilievo da .zip'),
            callback=self.run,
            parent=self.iface.mainWindow())

    def unload(self):
        """Rimuove gli elementi dell'interfaccia grafica."""
        for action in self.actions:
            self.iface.removePluginMenu(self.tr(u'&Collector Importer'), action)
            self.iface.removeToolBarIcon(action)
        del self.toolbar

    def run(self):
        """Funzione principale eseguita quando si clicca il pulsante."""
        zip_path, _ = QFileDialog.getOpenFileName(
            self.iface.mainWindow(), "Seleziona il file .zip del rilievo", "", "File ZIP (*.zip)")
        if not zip_path:
            return

        with tempfile.TemporaryDirectory() as temp_dir:
            try:
                with zipfile.ZipFile(zip_path, 'r') as zip_ref:
                    zip_ref.extractall(temp_dir)
            except Exception as e:
                self.iface.messageBar().pushMessage("Errore", f"Impossibile estrarre il file zip: {e}", level=Qgis.Critical, duration=5)
                return

            csv_path = os.path.join(temp_dir, 'rilievo.csv')
            if not os.path.exists(csv_path):
                self.iface.messageBar().pushMessage("Errore", "Il file 'rilievo.csv' non è stato trovato nello zip.", level=Qgis.Critical, duration=5)
                return

            try:
                with open(csv_path, 'r', encoding='utf-8-sig') as f:
                    f.readline()
                    reader = csv.reader(f)
                    headers = []
                    for row in reader:
                        if row:
                            headers = [h.strip() for h in row]
                            break
                
                style_fields = [h for h in headers if h.lower() not in ['latitudine_wgs84', 'longitudine_wgs84', 'e_lv95', 'n_lv95']]

                dialog = StyleDialog(style_fields)
                if dialog.exec_():
                    output_gpkg_path = dialog.get_output_path()
                    label_field = dialog.get_label_field()
                    category_field = dialog.get_category_field()
                    self.process_csv(csv_path, temp_dir, output_gpkg_path, label_field, category_field)

            except Exception as e:
                self.iface.messageBar().pushMessage("Errore", f"Impossibile leggere le intestazioni dal CSV: {e}", level=Qgis.Critical, duration=10)

    def process_csv(self, csv_path, temp_dir, output_gpkg_path, label_field, category_field):
        """Legge il CSV, copia le foto e crea un file GeoPackage."""
        try:
            # --- INIZIO NUOVA LOGICA PER FOTO E SALVATAGGIO ---
            output_dir = os.path.dirname(output_gpkg_path)
            permanent_photo_dir = os.path.join(output_dir, 'foto')
            temp_photo_dir = os.path.join(temp_dir, 'foto')
            
            # Crea la cartella permanente per le foto, se non esiste
            if os.path.exists(temp_photo_dir):
                os.makedirs(permanent_photo_dir, exist_ok=True)
            # --- FINE NUOVA LOGICA ---

            with open(csv_path, 'r', encoding='utf-8-sig') as f:
                f.readline()
                reader = csv.reader(f)
                headers = []
                for row in reader:
                    if row:
                        headers = row
                        break
                
                if not headers:
                    self.iface.messageBar().pushMessage("Errore", "Nessuna riga di intestazione trovata.", level=Qgis.Critical, duration=10)
                    return

                cleaned_headers = [h.strip() for h in headers]
                fields = QgsFields()
                fields.append(QgsField("id", QVariant.Int))
                for header in cleaned_headers:
                    if header.lower() not in ['latitudine_wgs84', 'longitudine_wgs84', 'e_lv95', 'n_lv95']:
                        fields.append(QgsField(header, QVariant.String))

                crs = QgsCoordinateReferenceSystem("EPSG:4326")
                
                # Prepara la scrittura del file GeoPackage
                writer = QgsVectorFileWriter.create(
                    output_gpkg_path,
                    fields,
                    QgsWkbTypes.Point,
                    crs,
                    QgsProject.instance().transformContext(),
                    QgsVectorFileWriter.SaveVectorOptions()
                )

                if writer.hasError() != QgsVectorFileWriter.NoError:
                    self.iface.messageBar().pushMessage("Errore", f"Creazione GeoPackage fallita: {writer.errorMessage()}", level=Qgis.Critical)
                    return

                lat_idx = cleaned_headers.index('latitudine_wgs84')
                lon_idx = cleaned_headers.index('longitudine_wgs84')

                feature_id = 1
                for row in reader:
                    try:
                        if not row or not row[lat_idx] or not row[lon_idx]: continue
                        lat, lon = float(row[lat_idx]), float(row[lon_idx])
                        feature = QgsFeature()
                        feature.setGeometry(QgsGeometry.fromPointXY(QgsPointXY(lon, lat)))
                        attributes = [feature_id]
                        for i, header in enumerate(cleaned_headers):
                            if header.lower() in ['latitudine_wgs84', 'longitudine_wgs84', 'e_lv95', 'n_lv95']: continue
                            value = row[i]
                            if header.lower() == 'foto' and value:
                                temp_photo_path = os.path.join(temp_photo_dir, value)
                                permanent_photo_path = os.path.join(permanent_photo_dir, value)
                                
                                # Copia la foto dalla cartella temp a quella permanente
                                if os.path.exists(temp_photo_path):
                                    shutil.copy2(temp_photo_path, permanent_photo_path)
                                
                                # Salva il percorso permanente e corretto
                                value = permanent_photo_path.replace('\\', '/')
                            attributes.append(value)
                        feature.setFields(fields) # Assicura che la feature abbia la struttura corretta
                        feature.setAttributes(attributes)
                        writer.addFeature(feature)
                        feature_id += 1
                    except (ValueError, IndexError): continue
                
                del writer # Finalizza la scrittura del file

                # Carica il nuovo layer in QGIS
                layer_name = os.path.basename(output_gpkg_path).replace('.gpkg', '')
                layer = QgsVectorLayer(output_gpkg_path, layer_name, "ogr")
                if not layer.isValid():
                    self.iface.messageBar().pushMessage("Errore", "Impossibile caricare il layer GeoPackage creato.", level=Qgis.Critical)
                    return
                
                QgsProject.instance().addMapLayer(layer)

                if label_field != "Nessuna":
                    self.apply_labeling(layer, label_field)
                if category_field != "Nessuna":
                    self.apply_categorized_renderer(layer, category_field)
                if layer.fields().indexOf("foto") != -1:
                    self.setup_photo_integration(layer, "foto")

                self.iface.messageBar().pushMessage("Successo", f"Layer '{layer.name()}' salvato e aggiunto alla mappa.", level=Qgis.Success, duration=5)

        except Exception as e:
            self.iface.messageBar().pushMessage("Errore", f"Errore durante l'elaborazione del CSV: {e}", level=Qgis.Critical, duration=10)

    def apply_labeling(self, layer, field_name):
        """Applica etichette semplici a un layer."""
        label_settings = QgsPalLayerSettings()
        label_settings.fieldName = field_name
        label_settings.enabled = True
        layer.setLabeling(QgsVectorLayerSimpleLabeling(label_settings))
        layer.setLabelsEnabled(True)
        layer.triggerRepaint()

    def apply_categorized_renderer(self, layer, field_name):
        """Applica un renderer categorizzato con colori casuali."""
        unique_values = layer.uniqueValues(layer.fields().indexOf(field_name))
        
        categories = []
        for value in unique_values:
            symbol = QgsSymbol.defaultSymbol(layer.geometryType())
            color = QColor.fromRgb(random.randint(0, 255), random.randint(0, 255), random.randint(0, 255))
            symbol.setColor(color)
            category = QgsRendererCategory(value, symbol, str(value))
            categories.append(category)
            
        renderer = QgsCategorizedSymbolRenderer(field_name, categories)
        layer.setRenderer(renderer)
        layer.triggerRepaint()

    def setup_photo_integration(self, layer, photo_field_name):
        """Configura l'azione per aprire le foto e i Map Tips per le anteprime."""
        action_text = f'[% "{photo_field_name}" %]'
        action = QgsAction(QgsAction.OpenUrl, 'Apri Foto', action_text)

        action_manager = layer.actions()
        action_manager.addAction(action)

        html_expression = f"""
        <div style="font-family: sans-serif;">
        <h3>Anteprima Foto</h3>
        <img src="file:///[% "{photo_field_name}" %]" width="300" />
        </div>
        """
        layer.setMapTipTemplate(html_expression)
